function [x_new, unfeas, prof] = bevModel(x, u, dyn, veh)
%bevModel
% Model of a BEV vehicle. Advances the simulation by one timestep.

% States
battSOC = x{1};

% Controls
GN = u{1};
brakeCmd = u{2};

% Mission information
vehSpd = dyn.vehSpd;
vehAcc = dyn.vehAcc;
vehDist = dyn.vehDist;

% Slope (rad)
vehSlope = atan(dyn.vehGrad/100); % rad

% Longitudinal dynamics
[wheelSpd, wheelTrq, vehPrf, whPrf] = vehDynamicsModel(vehSpd, vehAcc, vehSlope, veh.body, veh.wh);
vehPrf.vehDist = vehDist;

% Brakes
[fdTrqOut, brakeUnfeas, brkPrf] = brakeModel(brakeCmd, wheelTrq);

% Final drive
[fdSpd, fdTrq, fdPrf] = finalDriveModel(wheelSpd, fdTrqOut, veh.fd);

% Gearbox
[shaftSpd, shaftTrq, gbPrf] = gearboxModel(fdSpd, fdTrq, GN, veh.gb);

% E-machine
[emTrq, emElPwr, emUnfeas, emPrf] = emModel(shaftSpd, shaftTrq, veh.em);

% Accessories
loadPwr = emElPwr; % May add electrical accesories power here

% ESS
if isfield(veh, 'dynCh') & (dyn.eRoad == 1 & loadPwr >= 0)
    battPwr = loadPwr - veh.dynCh.eff*veh.dynCh.ratedPwr;
else
    battPwr = loadPwr;
end
[battSOC_new, battCurr, battUnfeas, battPrf] = battModel(battSOC, battPwr, veh.batt, veh.dt);

% Update states
x_new = {battSOC_new};

% Combine unfeasibilities
unfeas = emUnfeas | battUnfeas | brakeUnfeas;
vehPrf.spdUnfeas = emPrf.emSpdUnfeas;
vehPrf.accUnfeas = emPrf.emTrqUnfeas | battUnfeas;

% Pack time profiles
prof = mergeStructs(vehPrf, whPrf, fdPrf, gbPrf, emPrf, battPrf, brkPrf);
prof.time = dyn.time;
prof.unfeas = unfeas;
